/*+++

 track.c
 Example program for the IsoPod (see http://www.newmicros.com)

 Sonar Tracking System. TC0 = Trigger, TC1 = Echo, PWMA0 = Servo signal.
 Tested with Devantech SRF04 Ultrasonic Range Finder and Hitec HS-322 servo.
 Sonar is mounted on servo, which scans back and forth (MAXLEFT, MAXRIGHT).
 When sonar detects object in specified range (RMIN, RMAX), it "locks on".

 Peter F Gray (petegray@ieee.org)
 16Feb03

---*/

#include "scdefs.h"

/* SCI definitions */
#define SCI0BR  0x0F00
#define SCI0CR  0x0F01
#define SCI0SR  0x0F02
#define SCI0DR  0x0F03

/* PWM definitions */
#define FA1     0x0FA1
#define E00     0x0E00
#define E03     0x0E03
#define E05     0x0E05
#define E06     0x0E06
#define E07     0x0E07
#define E0D     0x0E0D
#define E0E     0x0E0E
#define E0F     0x0E0F

/* TIMER definitions */
#define D47 0x0D47
#define D4A 0x0D4A
#define D4D 0x0D4D
#define D4E 0x0D4E
#define D4F 0x0D4F

/* servo left and right max values */
#define MAXLEFT 0x02E4
#define MAXRIGHT 0x0B9A
/* servo incremental movement value */
#define BIGINC 10
/* ignore anything below min, above max ranges */
#define RMIN 6
#define RMAX 18
/* program readability definitions */
#define INRANGE 1
#define OUTRANGE 0
#define LR 1
#define RL 0

/* global var - servo position */
int p;

/* SCI I/O routines */
inscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0x3000)!=0x3000);
  *b = *a;
}

outsci (a,b)
int  *a;
unsigned char *b;
{
  int status;
  while (*b) {
    do status = *SCI0SR; while ((status&0xC000)!=0xC000);
    *a = *b;
    *b++;
  }
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
  *a = *b;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

void outdec (chan, val)
unsigned int *chan;
unsigned int val;
{
  unsigned int u,t;
  unsigned char c;
  t = 10000;
  do {
    u = 0;
    while (val>=t) { val -= t; u++; }
    c = '0' + u;
    outscichar (chan,&c);
    t /= 10;
  } while (t>0);
}

/* ping the sonar, return result in inches */
ping ()
{
  unsigned int result;
  *D47 = 0x000D;                              /* Trigger (TMRC0) high */
  *D4D = 0;                                   /* zero TMRC1 CNTR      */
  *D47 = 0x0005;                              /* Trigger (TMRC0) low  */
  *D4F = 0x0080;                              /* falling edge capture */
  *D4E = 0x7880;                              /* IP/16 TMRC1 input    */
  for (result=0; result<4000; result++);      /* brief pause */
  result = *D4A;                              /* get capture value */
  result /= 183;                              /* convert to inches */
  outdec (SCI0DR,result);
  outsci (SCI0DR," ");
  outdec (SCI0DR,p);
  outsci (SCI0DR,"\15");
  return result;
}

/* move servo to new position */
moveto (newpos)
int newpos;
{
  int status;
  *E06 = newpos;
  status = *E00;
  *E00 = 0x00C3;
}

/* track LR or RL, while in-range or out-of-range */
track (inc,cond)
int inc, cond;
{
  int r;
  if (cond==INRANGE) {        /* scan while in-range */
    do {
      if (inc==LR) p += BIGINC; else p -= BIGINC;
      moveto(p); r = ping();
    }
    while (r>RMIN && r<RMAX && p>=MAXLEFT && p<=MAXRIGHT);
  }
  else {                      /* scan while out-of-range */
    do {
      if (inc==LR) p += BIGINC; else p -= BIGINC;
      moveto(p); r = ping();
    }
    while ((r<=RMIN || r>=RMAX) && p>=MAXLEFT && p<=MAXRIGHT);
  }
  if (p<=MAXLEFT || p>=MAXRIGHT) return 0;      /* i.e. stop tracking */
  return 1;
}

/* main program */
main ()
{
  char c;
  int r, tracking;
  *SCI0BR = 130;
  *SCI0CR = 12;

  *FA1 = 0xF413;
  *E03 = 0x8000;
  *E05 = 0x61A8;
  *E0D = 0;
  *E0E = 0;
  *E0F = 0x000E;
  outsci (SCI0DR,"\15\nTracking System Ready\15\n");

  while (1) {           /* loop forever scanning L-R, then R-L    */
                        /* when object found, scan back-and-forth */
                        /* between object's edges                 */
    tracking = 0;
    for (p=MAXLEFT; p<MAXRIGHT; p += BIGINC) {
      moveto (p);
      r = ping();
      if (r>RMIN && r<RMAX) tracking = 1;  /* found left edge of object    */
      while (tracking) {
                                           /* scan object L-R              */
        tracking = track (LR,INRANGE);
                                           /* reverse, look for right edge */
        if (tracking) tracking = track (RL,OUTRANGE);
                                           /* scan object R-L              */
        if (tracking) tracking = track (RL,INRANGE);
                                           /* reverse, look for left edge  */
        if (tracking) tracking = track (LR,OUTRANGE);

      }
    }
    tracking = 0;
    for (p=MAXRIGHT; p>MAXLEFT; p -= BIGINC) {
      moveto (p);
      r = ping();
      outdec (SCI0DR,r);
      outsci (SCI0DR,"\15");
      if (r>RMIN && r<RMAX) tracking = 1;  /* found right edge of object   */
      while (tracking) {
                                           /* scan object R-L              */
        tracking = track (RL,INRANGE);
                                           /* reverse, look for left edge  */
        if (tracking) tracking = track (LR,OUTRANGE);
                                           /* scan object L-R              */
        if (tracking) tracking = track (LR,INRANGE);
                                           /* reverse, look for right edge */
        if (tracking) tracking = track (RL,OUTRANGE);

      }
    }
  }
}

