/*+++

 threads.c
 Multithreading example for NMIN-0803-H3 (see http://www.newmicros.com)

 Runs 2 threads + 'main'. Context switching via timer interrupt.
 Functions th1 and th2 (see code) are the threads.
 th1 turns green LED on, th2 turns it off.
 'main' displays "Running" message, and then alternates between turning the
 yellow LED on/off for 30 interrupts. Every 60 interrupts, a "." is output.

 Peter F Gray (petegray@ieee.org)
 6Feb03

---*/

#include "scdefs.h"

#define THMAX 3
int thcur;              /* current thread number */
                        /* thread info (thread 0 is 'main') */
int thpc[THMAX];        /* program counter */
int thpri[THMAX];       /* primary register */
int thsec[THMAX];       /* secondary register */
int thx0[THMAX];        /* X0 register */
int thn[THMAX];         /* N register */
int thbp[THMAX];        /* base pointer */
int thsp[THMAX];        /* stack pointer */
int thssp[THMAX];       /* stack pointer (s/w) */
int thsr[THMAX];        /* status register */
int icntr;              /* interrupt counter (general purpose) */

int globint;

/* GPIO registers */
#define PAPUR   0x0FB0
#define PADR    0x0FB1
#define PADDR   0x0FB2
#define PAPER   0x0FB3
#define PAIAR   0x0FB4
#define PAIENR  0x0FB5
#define PAIPOLR 0x0FB6
#define PAIPR   0x0FB7
#define PAIESR  0x0FB8

/* SCI registers */
#define SCI0BR  0x0F00
#define SCI0CR  0x0F01
#define SCI0SR  0x0F02
#define SCI0DR  0x0F03

/* Group 7 priority register (Timer D) */
#define ITCNG7  0x0E67
/* Interrupt Priority Register */
#define IPR     0xFFFB

#define TD0CMP1 0x0D60
#define TD0CMP2 0x0D61
#define TD0CAP  0x0D62
#define TD0LOAD 0x0D63
#define TD0HOLD 0x0D64
#define TD0CNTR 0x0D65
#define TD0CTRL 0x0D66
#define TD0SCR  0x0D67

/* general-purpose SCI I/O routines */
outsci (a,b)
int  *a;
unsigned char *b;
{
  int status;
  while (*b) {
    do status = *SCI0SR; while ((status&0xC000)!=0xC000);
    *a = *b;
    *b++;
  }
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

/* the thread context switcher */
interrupt myint1 ()
{
#asm
                        ; INTERRUPT (context switch)
                        ; save registers
  MOVE R1,R0            ; - secondary
  MOVE X0,Y0            ; - x0
  MOVE N,Y1             ; - N

  MOVE X:thcur,N        ; current thread #
  MOVE #thsr,R2         ; base address of thsr array
  MOVE X:(SP),R1        ; SR
  MOVE R1,X:(R2+N)      ; store SR in correct element

  MOVE #thpc,R2         ; base address of thpc array
  LEA (SP)-             ; dec SP to get ret address
  MOVE X:(SP),R1        ; PC
  MOVE R1,X:(R2+N)      ; store PC in correct element

  MOVE #thpri,R2        ; base address of thpri array
  NOP
  MOVE A,X:(R2+N)       ; store PRI in correct element
  MOVE #thsec,R2        ; base address of thsec array
  NOP
  MOVE R0,X:(R2+N)      ; store SEC in correct element
  MOVE #thx0,R2         ; base address of thx0 array
  NOP
  MOVE Y0,X:(R2+N)      ; store X0 in correct element
  MOVE #thn,R2          ; base address of thn array
  NOP
  MOVE Y1,X:(R2+N)      ; store N in correct element
  LEA (SP)-
  MOVE #thsp,R2         ; base address of thsp array
  NOP
  MOVE SP,X:(R2+N)      ; store SP in correct element

  MOVE X:(R3),R0        ; saved BP into R0
  MOVE #thbp,R2         ; base address of thbp array
  NOP
  MOVE R0,X:(R2+N)      ; store BP in correct element

  LEA (R3+1)            ; adjust (due to interrupt function frame)
  MOVE #thssp,R2        ; base address of thssp array
  NOP
  MOVE R3,X:(R2+N)      ; store SP (s/w) in correct element

                        ; current context is now saved

  MOVE X:icntr,A        ; increment the interrupt counter
  INC A
  CMP #$FFFF,A
  JNE NOICRST
  CLR A
NOICRST:
  MOVE A,X:icntr

  MOVE N,A              ; calculate next thread #
  INC A
  CMP #3,A              ; i.e. THMAX
  JNE NOTHRRST          ; don't reset thread # to zero
  CLR A
NOTHRRST:
  MOVE A1,N
  MOVE N,X:thcur        ; store current thread #

  MOVE #3431,R0         ; clear compare interrupt flag
  NOP
  MOVE X:(R0),X0        ; by ANDing TD0SCR with 7FFF
  MOVE #32767,A
  AND X0,A
  MOVE A1,X:(R0)

  MOVE #thsp,R2         ; switch to thread's stack
  NOP                   ; and reconstruct ready for RTI
  MOVE X:(R2+N),SP

  MOVE #thpc,R2         ; put new thread's PC on h/w stack
  LEA (SP)+             ; point to ret addr
  MOVE X:(R2+N),X0      ; get new ret addr
  NOP
  MOVE X0,X:(SP)        ; and store on h/w stack
  NOP
  LEA (SP)+
  MOVE #thsr,R2         ; SR onto h/w stack
  NOP
  MOVE X:(R2+N),SR

  MOVE #thssp,R2
  NOP
  MOVE X:(R2+N),R3      ; restore s/w SP
  MOVE #thpri,R2
  NOP

  MOVE X:(R2+N),A       ; restore primary
  MOVE #thsec,R2
  NOP
  MOVE X:(R2+N),R1      ; restore secondary
  MOVE #thx0,R2
  NOP
  MOVE X:(R2+N),X0      ; restore X0

  MOVE #thbp,R0
  NOP
  MOVE X:(R0+N),R2      ; restore BP

  MOVE #thn,R0
  NOP
  MOVE X:(R0+N),N       ; restore N

  RTI

#endasm

}

/* Thread #1 - turn green LED on */
th1() {
  while (1) {
    *PADR |= 0x0010;
  }
}

/* Thread #2 - turn green LED off */
th2() {
  while (1) {
    *PADR &= 0x00EF;
  }
}

main ()
{
  unsigned int  i;
  *SCI0BR = 260;                        /* set up SCI */
  *SCI0CR = 12;
  outsci (SCI0DR,"\15\nRunning 3 threads.\15\n");

                                        /* initialize threads */
  thcur = 0;
  for (i=0; i<THMAX; i++) {             /* basic thread info */
    thpc[i] = thpri[i] = thsec[i] = thx0[i] = thn[i] = 0;
    thssp[i] = 200-(30*i);              /* stack pointer (s/w) */
    thsp[i] = thssp[i];                 /* stack pointer */
    thbp[i] = thssp[i];                 /* base pointer */
  }
  icntr = 0;                            /* zero interrupt counter */

  thpc[1] = th1;                        /* thread addresses */
  thpc[2] = th2;

  *PAIAR = 0;                           /* set up GPIO */
  *PAIENR = 0;                          /* (for green/yellow LEDs) */
  *PAIPOLR = 0;
  *PAIESR = 0;
  *PAPER = 0x00C7;
  *PADDR = 0x0038;
  *PAPUR = 0x00FF;
  *PADR = 0;                            /* set to zero (i.e. 'off') */

                        /* set up timer */
  *ITCNG7 = 0x0700;     /* Group Priority for Timer D, channel 0 */
  *IPR = 0xFE00;        /* set Interrupt Priority Register */
  *TD0LOAD = 10000;     /* reload value (lower=faster interrupts) */
  *TD0CMP2 = 0x0000;    /* comparison value */
  *TD0SCR = 0x4000;     /* enable compare interrupt */
  *TD0CTRL = 0x3E30;    /* count IPBus/128, repeat, reinit, count down */

                        /* main prog, thread #0 */
  while (1) {
    if (icntr<30)
      *PADR &= 0x00DF;                  /* turn off for 30 interrupts */
    else
      *PADR |= 0x0020;                  /* turn on for 30 interrupts */
    if (icntr>60) outsci (SCI0DR,".");
    if (icntr>60) icntr = 0;            /* and reset to zero */
  }
}

