/*+++

 PWM/TIMER/SCI example program for the IsoPod (see http://www.newmicros.com)

 Allows the user to control Lynxmotion's L5 robot arm (see http://www.lynxmotion.com)
 via the PC (using hyperterm, minicom etc).

 Velocity control achieved by breaking each interrupt into chunks and only
 activating a particular servo for a percentage of each chunk, depending upon
 the setting (1-9) of the servo velocity.

 Peter F Gray (petegray@ieee.org)
 11Jan03

---*/

#include "scdefs.h"

/* SCI Registers */
#define SCI0BR  0x0F00
#define SCI0CR  0x0F01
#define SCI0SR  0x0F02
#define SCI0DR  0x0F03

#define FA1     0x0FA1
#define E00     0x0E00
#define E03     0x0E03
#define E05     0x0E05
#define E06     0x0E06
#define E07     0x0E07
#define E08     0x0E08
#define E09     0x0E09
#define E0A     0x0E0A
#define E0D     0x0E0D
#define E0E     0x0E0E
#define E0F     0x0E0F

/* Servo step increment */
#define SINC     20

/* Group 7 priority register (Timer D) */
#define ITCNG7  0x0E67
/* Interrupt Priority Register */
#define IPR     0xFFFB

/* Timer Registers */
#define TD0CMP1 0x0D60
#define TD0CMP2 0x0D61
#define TD0CAP  0x0D62
#define TD0LOAD 0x0D63
#define TD0HOLD 0x0D64
#define TD0CNTR 0x0D65
#define TD0CTRL 0x0D66
#define TD0SCR  0x0D67

/* current positions of each servo */
int base;
int sh;
int el;
int wr;
int gr;

/* new (target) positions of each servo */
int nbase;
int nsh;
int nel;
int nwr;
int ngr;

/* stored servo positions */
int pb[10];
int ps[10];
int pe[10];
int pw[10];
int pg[10];
int iflag;

/* IO Routines */
inscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0x3000)!=0x3000);
  *b = *a;
}

outsci (a,b)
int  *a;
unsigned char *b;
{
  int status;
  while (*b) {
    do status = *SCI0SR; while ((status&0xC000)!=0xC000);
    *a = *b;
    *b++;
  }
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
  *a = *b;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscihex (a,b)
int *a;
int b;
{
  int  c;
  int  r;
  char t[5];
  c = 3;
  while (c>=0) { 
    r = b&0x000F;
    if (r < 10)
      t[c] = '0'+r;
    else
      t[c] = 'A'+((r)-10);
    b >>= 4;
    b = b&0x0FFF;
    c--;
  }
  t[4] = 0;
  outsci (a,t); 
}

/* activate the PWM */
dopwm()
{
  int dum;
  dum = *E00;
  *E00 = 0x00C3;
}

/* set the PWM values */
setregs ()
{
  *E06 = base;
  *E07 = sh;
  *E08 = el;
  *E09 = wr;
  *E0A = gr;
}

/* starting position */
park ()
{
  base = 0x0753;
  sh = 0x0753;
  el = 0x0753;
  wr = 0x0753;
  gr = 0x0753;
  setregs();
  dopwm();
}

/* translate the 'store' key */
xlates (ch)
char ch;
{
  if (ch=='Q') return 0;
  if (ch=='W') return 1;
  if (ch=='E') return 2;
  if (ch=='R') return 3;
  if (ch=='T') return 4;
  if (ch=='Y') return 5;
  if (ch=='U') return 6;
  if (ch=='I') return 7;
  if (ch=='O') return 8;
  if (ch=='P') return 9;
  return 10;
}

/* translate the 'recall' key */
xlatep (ch)
char ch;
{
  if (ch=='1') return 0;
  if (ch=='2') return 1;
  if (ch=='3') return 2;
  if (ch=='4') return 3;
  if (ch=='5') return 4;
  if (ch=='6') return 5;
  if (ch=='7') return 6;
  if (ch=='8') return 7;
  if (ch=='9') return 8;
  if (ch=='0') return 9;
  return 10;
}

/* timer interrupt routine, sets the iflag */

interrupt myint1 ()
{
#asm
  LEA (R3-1)            ; INTERRUPT - save important registers
  MOVE A1,X:(R3)        ; primary
  LEA (R3-1)            ;
  MOVE R1,X:(R3)        ; secondary
  LEA (R3-1)            ;
  MOVE X0,X:(R3)        ;
#endasm

  *TD0SCR &= 0x7FFF;    /* clear compare interrupt flag */
  iflag = 1;            /* set the interrupt flag */

#asm
  MOVE X:(R3),X0        ; restore registers
  LEA (R3+1)            ;
  MOVE X:(R3),R1        ;
  LEA (R3+1)            ;
  MOVE X:(R3),A1        ;
  LEA (R3+1)            ;
#endasm
}

/* display the menu */
menu ()
{ 
  outsci (SCI0DR,"\15\nQ,W,E,R,T,Y,U,I,O,P = Store settings\15\n");
  outsci (SCI0DR,"1,2,3,4,5,6,7,8,9,0 = Position to stored settings\15\n");
  outsci (SCI0DR,"A,S=Grip  D,F=Wrist  G,H=Elbow  J,K=Shoulder  L,;=Base\15\n");
  outsci (SCI0DR,"V=Velocity Settings\15\n");
  outsci (SCI0DR,"SPACE=Show Settings, M=Menu (these instructions)\15\n");
}

main ()
{
  char c, c2, i1;
  int  i, v, working;
  int  vb[10];
  int  vs[10];
  int  ve[10];
  int  vw[10];
  int  vg[10];
  int  cvb, cvs, cve, cvw, cvg;

  *SCI0BR = 0x0082;             /* set up the modules */
  *SCI0CR = 12;
  *FA1 = 0xF413; 
  *E03 = 0x8000;
  *E05 = 0x61A8;
  *E0D = 0;
  *E0E = 0;
  *E0F = 0x000E;
  park();
  outsci (SCI0DR,"\15\nREADY\15\n");
  for (v=0; v<10; v++) {
    pb[v]=base; ps[v]=sh; pe[v]=el; pw[v]=wr; pg[v]=gr;
    vb[v]=9; vs[v]=9; ve[v]=9; vw[v]=9; vg[v]=9;
  }
  cvb = cvs = cve = cvw = cvg = 9;
  menu();

  *ITCNG7 = 0x0700;     /* Group Priority for Timer D, channel 0 */
  *IPR = 0xFE00;        /* set Interrupt Priority Register */

  *TD0LOAD = 75;        /* reload value */
  *TD0CMP2 = 0x0000;    /* comparison value */
  *TD0SCR = 0x4000;     /* enable compare interrupt */
  *TD0CTRL = 0x3E30;    /* count IPBus/128, repeat, reinit, count down */
  c = ' ';

loop:
  if (c==' ' || c=='M' || c=='V')
    outsci (SCI0DR,"\15\n>");
  inscichar (SCI0DR,&c);                /* input the command */
  
                                        /* inc/dec the position values */
  if (c=='A' && gr>0x02E4) gr -= SINC;
  if (c=='S' && gr<0x0B9A) gr += SINC;
  if (c=='L' && base<0x0B9A) base += SINC;
  if (c==';' && base>0x02E4) base -= SINC;
  if (c=='J' && sh>0x02E4) sh -= SINC;
  if (c=='K' && sh<0x0B9A) sh += SINC;
  if (c=='G' && el>0x02E4) el -= SINC;
  if (c=='H' && el<0x0B9A) el += SINC;
  if (c=='D' && wr<0x0B9A) wr += SINC;
  if (c=='F' && wr>0x02E4) wr -= SINC;

                                        /* display the current values */
  if (c==' ') {
    outsci (SCI0DR,"\15\nBase="); outscihex(SCI0DR,base);
    outsci (SCI0DR," Shoulder="); outscihex(SCI0DR,sh);
    outsci (SCI0DR," Elbow="); outscihex(SCI0DR,el);
    outsci (SCI0DR," Wrist="); outscihex(SCI0DR,wr);
    outsci (SCI0DR," Grip="); outscihex(SCI0DR,gr);
    outsci (SCI0DR,"\15\nVelocities=");
    for (i=0; i<5; i++) {
      if (i==0) v = cvb;
      if (i==1) v = cvs;
      if (i==2) v = cve;
      if (i==3) v = cvw;
      if (i==4) v = cvg;
      c2 = '0' + v;
      outscichar (SCI0DR,&c2);
      if (i<4) outsci (SCI0DR,",");
    }
    outsci (SCI0DR,"\15\n");
  }
  if (c=='M') menu();

                                        /* set the velocity of each servo */
  if (c=='V') {
    outsci (SCI0DR,"Velocities [1(slow)-9(fast)] BSEWG ");
    for (i=0; i<5; i++) {
      c2 = ' ';
      do inscichar (SCI0DR,&c2); while (c2<'1' || c2>'9');
      outscichar (SCI0DR,&c2);
      v = c2 - '0';
      if (i==0) cvb = v;
      if (i==1) cvs = v;
      if (i==2) cve = v;
      if (i==3) cvw = v;
      if (i==4) cvg = v;
    }
  }
  i1 = xlates(c);                       /* store requested */
  if (i1<10) {
    pb[i1] = base; ps[i1] = sh; pe[i1] = el; pw[i1] = wr; pg[i1] = gr;
    vb[i1] = cvb; vs[i1] = cvs; ve[i1] = cve; vw[i1] = cvw; vg[i1] = cvg;
    outsci (SCI0DR," Stored\15\n>");
  }
  i1 = xlatep(c);                       /* recall requested */
  if (i1<10) {
    working = 1;
    i = 1;
                                        /* set target values */

    nbase = pb[i1]; nsh = ps[i1]; nel = pe[i1]; nwr = pw[i1]; ngr = pg[i1];

    while (working==1) {                /* until we're done... */
      if (base<nbase && i<=vb[i1]) base++;
      if (base>nbase && i<=vb[i1]) base--;
      if (sh<nsh && i<=vs[i1]) sh++;
      if (sh>nsh && i<=vs[i1]) sh--;
      if (el<nel && i<=ve[i1]) el++;
      if (el>nel && i<=ve[i1]) el--;
      if (wr<nwr && i<=vw[i1]) wr++;
      if (wr>nwr && i<=vw[i1]) wr--;
      if (gr<ngr && i<=vg[i1]) gr++;
      if (gr>ngr && i<=vg[i1]) gr--;
      if (nbase==base && nsh==sh && nel==el && nwr==wr && ngr==gr) working=0;
      i++;
      if (i>9) i=1;
      setregs();                        /* set the PWM registers */
      dopwm();                          /* and activate */
      iflag=0;                          /* clear the interrupt flag */
      while(iflag==0);                  /* and wait for interrupt */
    }
  }
  setregs();                            /* set the PWM registers */
  dopwm();                              /* and activate */
  goto loop;                            /* go and prompt user */
}

