/*+++

 I2C example program for IsoPod (see http://www.newmicros.com)

 Performs I2C protocol communications via GPIO. PD0 = SDA, PD1 = SCL.
 LEDs (Port E) are turned on/off to reflect the state of the I2C bus.

 IsoPod is the I2C Master, standard mode, 7-bit addressing.

 Peter F Gray (petegray@ieee.org)
 5Jul03

---*/

#include "scdefs.h"
#include "gpio805d.h"
#include "gpio805e.h"
#include "i2cprot.h"

// SCI registers
#define SCI0BR  0x0F00
#define SCI0CR  0x0F01
#define SCI0SR  0x0F02
#define SCI0DR  0x0F03

// I/O routines for SCI (i.e. PC hyperterm or minicom communication)
inscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0x3000)!=0x3000);
  *b = *a;
}

outsci (a,b)
int  *a;
unsigned char *b;
{
  unsigned int status;
  while (*b) {
    do status = *SCI0SR; while ((status&0xC000)!=0xC000);
    *a = *b;
    *b++;
  }
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscichar (a,b)
int  *a;
unsigned char b;
{
  int status;

  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
  *a = b;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscihex (a,b)
int *a;
int b;
{
  int  c,r,neg;
  char t[5];

  c = 3; neg = 0;
  if ((b&0x8000)==0x8000) {                   /* test for sign bit */
    neg = 1;
    b &= 0x7FFF;                              /* remove if found */
    }
  while (c>=0) { 
    r = b&0x000F;
    if (r < 10)
      t[c] = '0'+r;
    else
      t[c] = 'A'+((r)-10);
    b >>= 4;
    c--;
    if (neg&&(c==0)) b |= 0x0008;             /* put sign bit back */
  }
  t[4] = 0;
  outsci (a,t); 
}

//--------------------------------------------------------------------------
// main program

main ()
{
  char c;
  int   result, p;

  *SCI0BR = 260;                        // set up SCI (9600, 8N1)
  *SCI0CR = 12;                         // and display instructions...

  outsci (SCI0DR,"\15\nI2C Master, SCI Module Ready\15\n");

  *PDIAR = 0;                           // set up GPIO - port D
  *PDIENR = 0;
  *PDIPOLR = 0;
  *PDIESR = 0;
  *PDPER = 0x00FC;                      // peripheral enable off
  *PDDDR = 0x0003;                      // configure for output
  *PDPUR = 0x00FF;                      // pull-ups enabled

  *PEIAR = 0;                           // set up GPIO - port E (LEDs)
  *PEIENR = 0;
  *PEIPOLR = 0;
  *PEIESR = 0;
  *PEPER = 0x00F3;                      // peripheral enable off
  *PEDDR = 0x000C;                      // configure for output
  *PEPUR = 0x00FF;                      // pull-ups enabled

  i2c_init();                           // init the I2C bus
  while (1) {                           // loop forever
    outsci (SCI0DR,"Press any key to continue\15\n");
    inscichar (SCI0DR,&c);              // wait for character input
    outsci (SCI0DR,"Start...\15\n");

//+++ Example - Reading the CMPS03 Compass (byte)

    if (i2c_start(0x00C0,I2CWRITE))
      outsci (SCI0DR,"No response\15\n");
    else
      if (i2c_snd_byte(1))
        outsci (SCI0DR,"No response to register request\15\n");
      else
        if (i2c_start(0x00C0,I2CREAD))
          outsci (SCI0DR,"No response read request\15\n");
        else {
          result = i2c_rcv_byte(I2CNOACK);
          outscihex (SCI0DR,result);
        }
    i2c_stop();

//---

//+++ Example - Reading the CMPS03 Compass (word)
/*
    if (i2c_start(0x00C0,I2CWRITE))
      outsci (SCI0DR,"No response\15\n");
    else
      if (i2c_snd_byte(1))
        outsci (SCI0DR,"No response register request\15\n");
      else
        if (i2c_start(0x00C0,I2CREAD))
          outsci (SCI0DR,"No response read request\15\n");
        else {
          result = i2c_rcv_byte(I2CACK);
          outscihex (SCI0DR,result);
          result = i2c_rcv_byte(I2CNOACK);
          outscihex (SCI0DR,result);
        }
    i2c_stop();
*/
//---

//+++ Example - Calibrating the CMPS03 Compass
/*
    for (p=1; p<5; p++) {
      if (p==1) outsci(SCI0DR,"North");
      if (p==2) outsci(SCI0DR,"East");
      if (p==3) outsci(SCI0DR,"South");
      if (p==4) outsci(SCI0DR,"West");
      outsci(SCI0DR,"\15\n");
      inscichar(SCI0DR,&c);
      if (i2c_start(0x00C0,I2CWRITE))
        outsci (SCI0DR,"No response\15\n");
      else
        if (i2c_snd_byte(15))
          outsci (SCI0DR,"No response register request\15\n");
        else
          if (i2c_snd_byte(255))
            outsci (SCI0DR,"No response write request\15\n");
      i2c_stop();
    }
*/
//---

    outsci (SCI0DR,"...End\15\n");
  }

}
