/*+++

 h803.c
 Example program for the NMIN-0803-H3 (see http://www.newmicros.com)

 Control motor speed using SCI (e.g. hyperterm) via H-Bridge

 PWMA0 = PWM Signal to H-Bridge, PA0 = Direction signal to H-Bridge,
 J18 (M1A/B) = Motor +/-, J21 (Vs,GND) = Power (5V).

 Tested with Portescap 22V28 Motor and R22 Planetary Gearhead.

 Peter F Gray (petegray@ieee.org)
 25Feb03

---*/

#include "scdefs.h"

/* SCI definitions */
#define SCI0BR  0x0F00
#define SCI0CR  0x0F01
#define SCI0SR  0x0F02
#define SCI0DR  0x0F03

/* PWM definitions */
#define FA1     0x0FA1
#define E00     0x0E00
#define E03     0x0E03
#define E05     0x0E05
#define E06     0x0E06
#define E07     0x0E07
#define E0D     0x0E0D
#define E0E     0x0E0E
#define E0F     0x0E0F

/* GPIO registers */
#define PAPUR   0x0FB0
#define PADR    0x0FB1
#define PADDR   0x0FB2
#define PAPER   0x0FB3
#define PAIAR   0x0FB4
#define PAIENR  0x0FB5
#define PAIPOLR 0x0FB6
#define PAIPR   0x0FB7
#define PAIESR  0x0FB8

/* servo incremental movement value */
#define BIGINC  200
#define HIGHVAL 0x6400
#define FORWARD 0
#define REVERSE 1

/* global var - motor speed */
int s;

/* SCI I/O routines */
inscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0x3000)!=0x3000);
  *b = *a;
}

outsci (a,b)
int  *a;
unsigned char *b;
{
  int status;
  while (*b) {
    do status = *SCI0SR; while ((status&0xC000)!=0xC000);
    *a = *b;
    *b++;
  }
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscichar (a,b)
int  *a;
unsigned char *b;
{
  int status;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
  *a = *b;
  do status = *SCI0SR; while ((status&0xC000)!=0xC000);
}

outscihex (a,b)
int *a;
int b;
{
  int  c,r,neg;
  char t[5];
  c = 3; neg = 0;
  if ((b&0x8000)==0x8000) {                   /* test for sign bit */
    neg = 1;
    b &= 0x7FFF;                              /* remove if found */
    }
  while (c>=0) { 
    r = b&0x000F;
    if (r < 10)
      t[c] = '0'+r;
    else
      t[c] = 'A'+((r)-10);
    b >>= 4;
    c--;
    if (neg&&(c==0)) b |= 0x0008;             /* put sign bit back */
  }
  t[4] = 0;
  outsci (a,t); 
}

/* set new servo speed */
servo (news)
int news;
{
  int status;
  *E06 = news;
  status = *E00;
  *E00 = 0x00C3;
}

/* main program */
main ()
{
  unsigned char ch;
  *SCI0BR = 130;                /* set up SCI registers */
  *SCI0CR = 12;                 /* 9600, 8N1 */

  *FA1 = 0xF413;                /* set up PWM registers */
  *E03 = 0x8000;
  *E05 = 0x61A8;
  *E0D = 0;
  *E0E = 0;
  *E0F = 0x000E;

  *PAIAR = 0;                   /* set up GPIO */
  *PAIENR = 0;
  *PAIPOLR = 0;
  *PAIESR = 0;
  *PAPER = 0x00FE;
  *PADDR = 0x0001;
  *PAPUR = 0x00FF;

  outsci (SCI0DR,"\15\nSystem Ready. F=Fast, S=Slow, R=Reverse.\15\n");

  s = 0;                        /* set speed = 0 */
  *PADR = FORWARD;              /* set direction */
  servo (s);                    /* set motor speed */

  while (1) {                   /* loop forever */
    outscihex (SCI0DR,s);
    outsci (SCI0DR," ");
    if (*PADR==FORWARD) outsci (SCI0DR,"For"); else outsci (SCI0DR,"Rev");
    outsci (SCI0DR,"\15");
    inscichar (SCI0DR,&ch);
    if (ch=='f' || ch=='F') s += BIGINC;
    if (ch=='s' || ch=='S') s -= BIGINC;
    if (ch=='r' || ch=='R') {
      if (*PADR==FORWARD) *PADR = REVERSE; else *PADR = FORWARD;
    }
    if (s>HIGHVAL) s = HIGHVAL; 
    if (s<0) s = 0; 
    if (*PADR==FORWARD)         /* set new motor speed */
      servo (s);                 
    else
      servo (HIGHVAL-s);
  }
}

